/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_init.c,v 1.80 2006/10/23 06:32:20 loic Exp $";
 
#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx__lib_types.h"
#include "mx__internals.h"
#include "mx__lib.h"
#include "mx__debug_dump.h"
#include "mx__error.h"
#include "mx__shmem.h"
#include "mx_stbar.h"
#include "mx__driver_interface.h"
#include "mx_version.h"
#include "mx_io.h"
#include "mx_cpu.h"

#ifdef MX_KERNEL
#define MX_VERBOSE_DEFAULT MX_DEBUG
#else
#define MX_VERBOSE_DEFAULT 0
#endif

#ifdef MX_KERNEL
#define MX_ZOMBIE_SEND_DEFAULT 0
#else
#define MX_ZOMBIE_SEND_DEFAULT 1
#endif

static int
parse_opt(const char *name, int dflt)
{
#ifndef MX_KERNEL
  const char *s;
  if ((s = getenv(name))) {
    switch (s[0]) {
    case 0: case 'n': case 'N':
      return 0;
    case 'y': case 'Y':
      return 1;
    default:
      if (s[0] >= '0' && s[0] <= '9') 
	return mx_strtol(s,0,0);
      mx_printf("MX:invalid env variable %s=%s\n", name,s);
    }
  }
#endif
  return dflt;
}

static inline mx_return_t
mx__check_driver_version(void)
{
#ifndef MX_KERNEL
  mx_get_version_t x;
  mx_endpt_handle_t handle;
  mx_return_t ret;

  if (mx__opt.no_myrinet)
    return MX_SUCCESS;

  ret = mx_open_any_board(&handle);
  if (ret != MX_SUCCESS)
    return mx__error_noep("mx_init:querying driver", ret);

  ret = mx__get_version(handle,&x);
  if (ret != MX_SUCCESS)
    return mx__error_noep("mx_init:querying driver for version info", ret);

  mx__close(handle);

  if (x.driver_api_magic / 256 != MX_DRIVER_API_MAGIC /256) {
    mx_printf("MX:driver-api-seq-num differ (lib=%d.%d,kernel=%d.%d)\n"
	      "\tMX Lib Version=%s\n"
	      "\tMX Lib Build=%s\n"
	      "\tMX Kernel Version=%s\n"
	      "\tMX Kernel Build=%s\n",
	      MX_DRIVER_API_MAGIC / 256, MX_DRIVER_API_MAGIC % 256,
	      x.driver_api_magic / 256, x.driver_api_magic % 256,
	      MX_VERSION_STR,
	      MX_BUILD_STR,
	      x.version_str,
	      x.build_str);
    return mx__error_noep("mx_init: driver is incompatible with library version",
			  MX_BAD_KERNEL_VERSION);
  }
#endif /* ~MX_KERNEL */

  return MX_SUCCESS;
}

static int mx__lib_api = MX_API;

MX_FUNC(mx_return_t)
mx__init_api(int app_api)
{
  mx_return_t ret = MX_SUCCESS;
  MX__MUTEX_LOCK(MX_LOCK);

  if (Mx_init_count) {
#ifndef MX_KERNEL
    ret = mx__error_noep("mx_init", MX_ALREADY_INITIALIZED);
#else
    /* makes no sense to return MX_ALREADY_INITIALIZED in the kernel lib,
     * just account the users and leave */
    Mx_init_count++;
#endif
    goto out;
  }

  Mx_init_count++;

#ifndef MX_KERNEL
  if (parse_opt("MX_LINEBUF", 0))
    setvbuf(stdout, NULL,  _IOLBF, BUFSIZ);
  mx__opt.cpus = parse_opt("MX_CPUS", 0);
#endif

#if MX_RUNTIME_OPT
  mx__opt.monothread = parse_opt("MX_MONOTHREAD", 0);
  mx__opt.verbose = parse_opt("MX_VERBOSE", MX_VERBOSE_DEFAULT);
  mx__opt.verbose_errors = parse_opt("MX_ERRORS_VERBOSE", mx__opt.verbose);
  mx__opt.errors_are_fatal = parse_opt("MX_ERRORS_ARE_FATAL", 1);
  mx__opt.sigusr1_handler = parse_opt("MX_DEBUG_SIGNAL_HANDLER", 0);
  mx__opt.disable_self = parse_opt("MX_DISABLE_SELF", 0);
  mx__opt.disable_shmem = parse_opt("MX_DISABLE_SHMEM", 0);
  mx__opt.rcache = parse_opt("MX_RCACHE", 0);
  mx__opt.coredump = parse_opt("MX_COREDUMP", 0);
  mx__opt.abort_sleeps = parse_opt("MX_ABORT_SLEEPS", 0);
  mx__opt.no_myrinet = parse_opt("MX_NO_MYRINET", 0);
  mx__opt.csum = parse_opt("MX_CSUM", 0);
  mx__opt.wc_fence = parse_opt("MX_WC_FENCE", 1);
  mx__opt.stats = parse_opt("MX_STATS", 0);
  mx__opt.fw_ack = parse_opt("MX_FW_ACK", 0) ? MX__REQUEST_STATE_ACKED : 0;
  mx__opt.pipeline_log = parse_opt("MX_PIPELINE_LOG", -1);
  mx__opt.zombie_send = parse_opt("MX_ZOMBIE_SEND", MX_ZOMBIE_SEND_DEFAULT);
  mx__opt.max_retries = parse_opt("MX_MAX_RETRIES", 1000);
  mx__opt.matter_debug = parse_opt("MX_MATTER_DEBUG", 1);
  mx__opt.imm_ack = parse_opt("MX_IMM_ACK", 4);
#if (MX_CPU_x86 || MX_CPU_x86_64) && defined __GNUC__
  mx__opt.intel_cpu = parse_opt("MX_INTEL_CPU", mx__cpu_is_intel());
#endif
#endif /* MX_RUNTIME_OPT */

  if (!MX_DEBUG_CSUM && mx__opt.csum) {
    mx_printf("Warning: checksum option ignored without --enable-debug\n");
  } else if (mx__opt.csum && MX_ONE_SIDED) {
    mx_printf("CSUM can't be enabled with --enable-one-sided\n");
    mx__opt.csum = 0;
  } else if (mx__opt.csum) {
    mx_printf("CSUM enabled\n");
  }

  if (MX_OS_UDRV) {
    mx__opt.monothread = 1;
    mx__opt.disable_shmem = 1;
    mx__opt.disable_self = 1;
  }

  if (mx__opt.no_myrinet)
    mx__opt.fw_ack = MX__REQUEST_STATE_ACKED;

  mx__init_signal_handling();
  mx__error_init();

#if !MX_OS_WINNT && !defined MX_KERNEL
  if (mx__opt.coredump) {
    struct rlimit c;
    c.rlim_cur = ~0L;
    c.rlim_max = ~0L;
    setrlimit(RLIMIT_CORE,&c);
  }
#endif

#if MX_CPU_x86 && defined __GNUC__ && MX_ENABLE_SSE2 && !defined MX_KERNEL
  if (!mx__cpu_has_sse2()) { 
    mx_printf("MX:mx_init: processor without sse2\n");
    mx_printf("MX:Please recompile using --disable-sse2\n");
    ret = mx__error_noep("mx_init:cannot run on this processor", MX_FAILURE);
    Mx_init_count = 0;
    goto out;
  }
#endif

  if (app_api >> 8 != mx__lib_api >> 8) {
    ret = mx__error_noep("mx_init: myriexpress.h used at compilation incompatible"
			 " with lib used at link time, api-seq-num differ (lib=%d,app=%d)",
			 MX_BAD_LIB_VERSION, mx__lib_api >> 8, app_api >> 8);
    Mx_init_count = 0;
    goto out;
  }

  ret = mx__check_driver_version();
  if (ret != MX_SUCCESS) {
    Mx_init_count = 0;
    goto out;
  }

 out:
  MX__MUTEX_UNLOCK(MX_LOCK);
  return ret;
}

#ifdef MX_VERSIONED_SYMS
mx_return_t
mx_init_v1(void)
{
  /* no way to detect an inconsistency with previous lib */
  return mx__init_api(MX_API);
}
__asm__(".symver mx_init_v1,mx_init@MX_0.0");
#endif
